/*****************************************************************************
**+------------------------------------------------------------------------+**
**|                                                                        |**
**|                Copyright 2010 Mistral Solutions Pvt Ltd.               |**
**|                                                                        |**
**|                                                                        |**
**|                                                                        |**   
**| This program is free software; you can redistribute it and/or          |**
**| modify it under the terms of the GNU General Public License as         |**
**| published by the Free Software Foundation; either version 2 of         |**
**| the License, or (at your option) any later version.                    |**
**|                                                                        |**
**| This program is distributed in the hope that it will be useful,        |**
**| but WITHOUT ANY WARRANTY; without even the implied warranty of         |**
**| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the           |**
**| GNU General Public License for more details.                           |**
**|                                                                        |**      
**| You should have received a copy of the GNU General Public License      |**
**| along with this program; if not, write to the Free Software            |**
**| Foundation, Inc., 59 Temple Place, Suite 330, Boston,                  |**
**| MA 02111-1307 USA                                                      |**
**+------------------------------------------------------------------------+**
*****************************************************************************/  

#include "DM814x_EVM.h"
#include "DM814x_types.h"

#define LISTLENGTH       (4)	// Max Command Header Per Port is 32
#define WRITE_CMD_SLOT   (0)	// Use command slot 0 for write. <= LISTLENGTH-1
#define READ_CMD_SLOT    (1)	// Use command slot 1 for read.  <= LISTLENGTH-1
#define PRDLENGTH        (2)   	// PRD Length = 2.
#define DATABUFFERLEN (  256) 	// DMA Data Buffer Length = 256

/* ------------------------------------------------------------------------------------------ *
 *                                                                                            *
 *  Command List Header									      *
 *											      *
 *  Maximum of 32 commands slots per port exist where each command occupies 8 DWs (64 Bytes). *
 *  The structure 'CmdListHeader' defines a single command header definition.                 *
 *  The start of the first Command List &CmdListHeader[0] needs to be programmed onto P0CLB.  *
 *  Command List Base Address should be 1K Byte Aligned.                                      *
 *                                                                                            *
 * ------------------------------------------------------------------------------------------ */
typedef struct {
	UINT32 CmdLen:5;   // bits[4:0]
	UINT32 Atapi:1;    // bit[5]
	UINT32 Write:1;    // bit[6]
	UINT32 Prefetch:1; // bit[7]
	UINT32 Reset:1;    // bit[8]
	UINT32 Bist:1;     // bit[9]
	UINT32 Rok:1;      // bit[10]
	UINT32 Rsv:1;      // bit[11]
	UINT32 Pmp:4;      // bits[15:12]
	UINT32 Prdtl:16;   // bits[31:16]
}CmdListHeaderW0;

typedef struct {
	UINT32 PrdByteCnt;      // bits[31:0]
}CmdListHeaderW1;

typedef struct {
	UINT32 CmdTableAddLow;  // bits[31:7]
}CmdListHeaderW2;

typedef struct {
	UINT32 CmdTableAddHigh; //bits[31:0]
}CmdListHeaderW3;

/* Command List Header */
typedef struct {
	CmdListHeaderW0 DW0;		// 4 Byte
	CmdListHeaderW1 DW1;		// 4 Byte
	CmdListHeaderW2 DW2;		// 4 Byte
	CmdListHeaderW3 DW3;		// 4 Byte
	UINT32          DW4;		// 4 Byte
	UINT32          DW5;		// 4 Byte
	UINT32          DW6;		// 4 Byte
	UINT32          DW7;		// 4 Byte
} CmdListHeader;		        // = 64 Byte


/* -------------------------------------------------------------------- *
 *									*
 *  Command Table Data Structure					*
 *									*
 *  Members: Command FIS						*
 *           ATAPI Command						*
 *           PRD Table							*
 *  Host to Device FIS Definition also named as Command FIS		*
 *  in AHCI Specification.						*
 *  MCommand Table requires alignment of 128 Bytes.			*
 * -------------------------------------------------------------------- */

//---------------------------Command FIS---------------------------//

typedef struct {
	UINT32 B0FisType:8; // bits[7:0]
	UINT32 BYTE1:8;     // bits[15:8]
	UINT32 B2Cmd:8;     // bits[23:16]
	UINT32 B3Feature:8; // bits[31:24]
}CmdFisWord0;

typedef struct {
	UINT32 B0LbaLow:8; // bits[7:0]
	UINT32 B1LbaMid:8; // bits[15:8]
	UINT32 B2LbaHigh:8;// bits[23:16]
	UINT32 B3Device:8; // bits[31:24]
}CmdFisWord1;

typedef struct {
	UINT32 B0LbaLowExp:8;  // bits[7:0]
	UINT32 B1LbaMidExp:8;  // bits[15:8]
	UINT32 B2LbaHighExp:8; // bits[23:16]
	UINT32 B3FeatureExp:8; // bits[31:24]
}CmdFisWord2;

typedef struct {
	UINT32 B0SecCnt:8;    // bits[7:0]
	UINT32 B1SecCntExp:8; // bits[15:8]
	UINT32 B2Rsv:8;       // bits[23:16]
	UINT32 B3Control:8;   // bits[31:24]
}CmdFisWord3;


typedef struct {
	UINT32 DWResv;        // bits[31:0]
}CmdFisWord4;

typedef struct {
	CmdFisWord0 DW0;		// 16 Byte
	CmdFisWord1 DW1;		// 16 Byte
	CmdFisWord2 DW2;		// 16 Byte
	CmdFisWord3 DW3;		// 16 Byte
	CmdFisWord4 DW4;		// 4 Byte
	UINT32      DW5;		// 4 Byte
	UINT32      DW6;		// 4 Byte
	UINT32      DW7;		// 4 Byte
	UINT32      DW8;		// 4 Byte
	UINT32      DW9;		// 4 Byte
	UINT32      DW10;		// 4 Byte
	UINT32      DW11;		// 4 Byte
	UINT32      DW12;		// 4 Byte
	UINT32      DW13;		// 4 Byte
	UINT32      DW14;		// 4 Byte
	UINT32      DW15;		// 4 Byte
}CommandFIS;		        // 112 Byte

//-------------------------Command FIS End-------------------------//
//--------------------------ATAPI Command -------------------------//

// ATAPI Command Data Structure
typedef struct {
	UINT32 ATAPI[4];
}Atapi;
//------------------------ATAPI Command End-----------------------//
//------------------------------PRDT -----------------------------//
// Physical Region Descriptor Table Data Structure
typedef struct {
	UINT32 DbaLow;      // bits[31:0]
}DbaAddressLow;

typedef struct {
	UINT32 DbaHigh;     // bits[31:0]
}DbaAddressHigh;

typedef struct {
	UINT32 DW2Reserved; // bits[31:0]
}PrdtRsv;

typedef struct {
	UINT32 DataBC:22;   // bits[21:0]
}DataByteCnt;


typedef struct {
	DbaAddressLow  DW0;
	DbaAddressHigh DW1;
	PrdtRsv        DW2;
	DataByteCnt    DW3;
}PRDT;
//----------------------------PRDT End---------------------------//

/******************** Command Table Data Structure ****************/
typedef struct {
	CommandFIS cfis;
	Atapi      atapi;
	UINT32     Rsv[12];
	PRDT       prdTable[8];	// Size set to 8 in to meet the minimum size for Command Table.
}CommandTable;

/****************** Command Table Data Structure end ***************/

/* -------------------------------------------------------------------- *
 *									*
 *  RECEIVE FIS Data Structure						*
 *									*
 *  Members: DMA Setup FIS (DSFIS)					*
 *           PIO Setup FIS (PSFIS)					*
 *           D2H Register FIS (RFIS)					*
 *           Set Device Bits FIS (SDBFIS)				*
 *           Unknown FIS (UFIS)						*
 *  Receive FIS requires the Receive FIS to be 256 byte aligned.	*
 * -------------------------------------------------------------------- */
//-------------------------DMA Setup FIS-------------------------//

typedef struct {
	UINT32 B0FisType:8;   // bits[7:0]
	UINT32 BYTE1:8;       // bits[15:8]
	UINT32 B2Rsv:8;       // bits[23:16]
	UINT32 B3Rsv:8;       // bits[31:24]
}DsfisW0;
/* DMA Setup FIS */
typedef struct {
	DsfisW0 DW0;				// 4 x 4	= 16 Byte
	UINT32 DW1DmaBuffLow;		// 4 Byte
	UINT32 DW2DmaBuffHigh;		// 4 Byte
	UINT32 DW3Rsv;				// 4 Byte
	UINT32 DW4DmaBuffOffset;	// 4 Byte
	UINT32 DW5DmaXfrCnt;		// 4 Byte
	UINT32 DW6Rsv;				// 4 Byte
}DMASetupFis;					// = 40Byte
//-----------------------DMA Setup FIS End-----------------------//
//-------------------------PIO Setup FIS ------------------------//

typedef struct {
	UINT32 B0FisType:8;  // bits[7:0]
	UINT32 BYTE1:8;      // bits[15:8]
	UINT32 B2Status:8;   // bits[23:16]
	UINT32 B3Errror:8;   // bits[31:24]
}PioSetupDW0;			 // 16 Byte

typedef struct {
	UINT32 B0LbaLow:8;  // bits[7:0]
	UINT32 B1LbaMid:8;  // bits[15:8]
	UINT32 B2LbaHigh:8; // bits[23:16]
	UINT32 B3Device:8;  // bits[31:24]
}PioSetupDW1;

typedef struct {
	UINT32 B0LbaLowExp:8; // bits[7:0]
	UINT32 B1LbaMidExp:8; // bits[15:8]
	UINT32 B2LbaHighExp:8;// bits[23:16]
	UINT32 B3Rsv:8;       // bits[31:24]
}PioSetupDW2;


typedef struct {
	UINT32 B0SecCnt:8;    // bits[7:0]
	UINT32 B1SecCntExp:8; // bits[15:8]
	UINT32 B2Rsv:8;       // bits[23:16]
	UINT32 B3Estatus:8;   // bits[31:24]
}PioSetupDW3;

typedef struct {
	UINT32 HW0XferCnt:16; // bits[15:0]
	UINT32 HW1Rsv:16;     // bits[31:16]
}PioSetupDW4;
/* PIO Setup FIS */
typedef struct {
	PioSetupDW0 DW0;	// 16 Byte
	PioSetupDW1 DW1;	// 16 Byte
	PioSetupDW2 DW2;	// 16 Byte
	PioSetupDW3 DW3;	// 16 Byte
	PioSetupDW4 DW4;	// 16 Byte
}PIOSetupFis;			// 80 Byte
//-----------------------PIO Setup FIS End ----------------------//
//--------------------------D2H Reg FIS--------------------------//

typedef struct {
	UINT32 B0FisType:8; // bits[7:0]
	UINT32 BYTE1:8;     // bits[15:8]
	UINT32 B2Status:8;  // bits[23:16]
	UINT32 B3Errror:8;  // bits[31:24]
}D2HRegDW0;

typedef struct {
	UINT32 B0LbaLow:8;  // bits[7:0]
	UINT32 B1LbaMid:8;  // bits[15:8]
	UINT32 B2LbaHigh:8; // bits[23:16]
	UINT32 B3Device:8;  // bits[31:24]
}D2HRegDW1;

typedef struct {
	UINT32 B0LbaLowExp:8;  // bits[7:0]
	UINT32 B1LbaMidExp:8;  // bits[15:8]
	UINT32 B2LbaHighExp:8; // bits[23:16]
	UINT32 B3Rsv:8;        // bits[31:24]
}D2HRegDW2;


typedef struct {
	UINT32 B0SecCnt:8;    // bits[7:0]
	UINT32 B1SecCntExp:8; // bits[15:8]
	UINT32 HW1Rsv:16;     // bits[31:16]
}D2HRegDW3;

typedef struct {
	UINT32 W0Rsv;         // bits[31:0]
}D2HRegDW4;

/* D2H Reg FIS */
typedef struct {
	D2HRegDW0 DW0;	// 16 Byte
	D2HRegDW1 DW1;	// 16 Byte
	D2HRegDW2 DW2;	// 16 Byte
	D2HRegDW3 DW3;	// 12 Byte
	D2HRegDW4 DW4;	//  4 Byte
}D2HRegFis;	        // 64 Byte
//------------------------D2H Reg FIS End------------------------//
//----------------------Set Device Bits FIS----------------------//

typedef struct {
	UINT32 B0FisType:8; // bits[7:0]
	UINT32 BYTE1:8;     // bits[15:8]
	UINT32 B2Status:8;  // bits[23:16]
	UINT32 B3Errror:8;  // bits[31:24]
}SetDevBitsDW0;

typedef struct {
	UINT32 SActive;     // bits[31:0]
}SetDevBitsDW1;

/* Set Device Bits FIS */
typedef struct {
	SetDevBitsDW0 DW0;	// 4 Byte
	SetDevBitsDW1 DW1;	// 4 Byte
}SetDevBitsFis;	        // 8 Byte
//----------------------Set Device Bits FIS----------------------//
//------------------------- Unkonwn FIS -------------------------//

typedef struct {
	UINT32 UserDefined; //bits[31:0]
}UnknownDWx;


typedef struct {
	UnknownDWx DW[16]; // 16 Words (Max 64 Bytes allowed)
}UnknownFis;           // 64 Byte

//----------------------- Unkonwn FIS End -----------------------//

//*************** Receive Register FIS Structure ****************//
typedef struct {
	DMASetupFis   DSFIS;		// 40B
	UINT32        Rsv1;			// 4B
	PIOSetupFis   PSFIS;		// 80B
	UINT32        Rsv2[3];
	D2HRegFis     RFIS;
	UINT32        Rsv3;
	SetDevBitsFis SDBFIS;
	UnknownFis    UFIS;
}ReceiveFis;

//************* Receive Register FIS Structure End **************//

/* ------------------------------------------------------------------------ *
 *                                                                          *
 *  Variable Definitions                                                    *
 *                                                                          *
 * ------------------------------------------------------------------------ */

extern unsigned char prdTableDataBuff[LISTLENGTH][PRDLENGTH][DATABUFFERLEN];
